/************************************************************************
* \file: trace_events.c
*
* \version: $Id: trace_events.c,v 1.9 2012/05/28 10:30:37 jayanth.mc Exp $
*
* This file implements event functionality in Linux 
*
* \component: Gen2 Trace
*
* \author   Sakthivelu S.  sakthivelu.s(o)in.bosch.com
*           A.Pape, *For flag implementation (later modified for Linux) *
*           T.Kuehn *For flag implementation (later modified for Linux) *
* \modified:Arun V
* \copyright: (c) 2003 - 2009 ADIT
*
***********************************************************************/

#include "trace_base.h"
#include "semaphore.h"

/**
* Set flag event pattern.
*
* \param CurData         Pointer to flag struct.
* \param EventPattern    The event bit pattern.
* \param SetMode         SetMode: 1 set, 0 clear.
*
* \return E_OK
*
*/
LOCAL S32 TRACE_set_FlagEvent_nolock(TRACE_flg* CurData, U32 EventPattern, BOOL SetMode)
{
    if( SetMode )
    {
        CurData->iflgptn |= EventPattern ;
    }
    else
    {
        CurData->iflgptn &= (~EventPattern);
    }
    return E_OK;
}

LOCAL S32 TRACE_set_FlagEvent(TRACE_flg* CurData, U32 EventPattern, BOOL SetMode)
{
    ER err = 0;
    (void)pthread_mutex_lock(&(CurData->EventMutex));
    err = TRACE_set_FlagEvent_nolock(CurData, EventPattern,SetMode);
    (void)pthread_mutex_unlock(&(CurData->EventMutex));
    return err;
}

/**
* Init flag attribute.
*
* \param NewFlag         Pointer to TRACE_flg structure.
*
* \return E_OK
*
*/

LOCAL ER TRACE_flag_AttrInit(TRACE_flg* NewFlag )
{
    S32 err = E_OK;
    pthread_condattr_t condAttr;
    pthread_mutexattr_t mutexAttr;

    if(NewFlag != NULL)
    {
      NewFlag->CondVarInitOk = FALSE;
      NewFlag->EventMutexInitOk = FALSE;
    }
    else
    {
      err = E_FAIL;
    }

    /* create cond attr and init cond variable */
    if(E_OK == err)
    {
      err = pthread_condattr_init(&condAttr);
      if(E_OK == err)
      {
        err = pthread_condattr_setpshared(&condAttr,PTHREAD_PROCESS_SHARED);
        if((E_OK == err) && (NULL != NewFlag))
        {
          err = pthread_cond_init(&(NewFlag->CondVar), &condAttr);
        }
        if((E_OK == err) && (NULL != NewFlag))
        {
          NewFlag->CondVarInitOk = TRUE;
        }
        /* destroy the attribute object */
        pthread_condattr_destroy(&condAttr);
      }
    }

    /* create mutex attr and init nutex variable */
    if(E_OK == err)
    {
      err = pthread_mutexattr_init(&mutexAttr);
      if(E_OK == err)
      {
	      err = pthread_mutexattr_setpshared(&mutexAttr,PTHREAD_PROCESS_SHARED);
        if((E_OK == err) && (NULL != NewFlag) )
        {
          err = pthread_mutex_init( &(NewFlag->EventMutex), &mutexAttr);
        }
        if( (E_OK == err) && (NULL != NewFlag) )
        {
          NewFlag->EventMutexInitOk = TRUE;
        }
        /* destroy the attribute object */
        pthread_mutexattr_destroy(&mutexAttr);
      }
    }
    return err;
}

/**
* Create a flag for events.
* Delete a flag.
*
* /param flgid         the current flag ID.
*
* /return E_OK
*         E_FAIL        Any error.
*/
EXPORT ER TRACE_delete_flag(TRACE_flg* pCover)
{
    S32 err = 0;
    ER ret = E_OK;

    if( pCover->CondVarInitOk )
    {
        err = pthread_cond_destroy( &(pCover->CondVar) );
        if ( err !=0 )
        {
          ret = E_FAIL;
        }
        else
        {
          pCover->CondVarInitOk = FALSE;
        }
    }
    if( pCover->EventMutexInitOk )
    {
        err = pthread_mutex_destroy( &(pCover->EventMutex) );
        if ( err !=0 )
        {
            ret = E_FAIL;
        }
        else
        {
            pCover->EventMutexInitOk = FALSE;
        }
    }
    return ret;
}

/**
*
* /param NewFlag         Pointer to flag struct.
*
* /return E_OK
*         E_FAIL          if error.
*/
EXPORT ER TRACE_create_flag(TRACE_flg* NewFlag)
{
    S32 err=0;

    if(NewFlag == NULL)
    {
        return E_FAIL;
    }
    NewFlag->iflgptn = 0;
    /* create flag */
    err = TRACE_flag_AttrInit( NewFlag );
    if( err != E_OK )
    {
        TRACE_delete_flag( NewFlag);
        return E_FAIL;
    }
    return E_OK;
}




/**
* Set a flag (bitpattern).
*  
* /param pCover        pointer to flag structure.
* /param setptn        flag pattern
*
* /return E_OK
*         E_FAIL       Any error.
*/
EXPORT ER  TRACE_set_flag(TRACE_flg* pCover, U32 setptn )
{
    S32 err = E_FAIL;
    if( pCover != NULL )
    {
      /* set event */
      err = TRACE_set_FlagEvent(pCover,setptn,TRUE);
      if(err == E_OK)
      { /*TRACE_set_FlagEvent() shall also included in the below critical section before call
         * calling pthread_cond_broadcast(). TRACE_set_FlagEvent_nolock() has to be called in this
         * case instead of TRACE_set_FlagEvent() */

        /* SWGIII-6060 FIX: Call pthread_cond_broadcast inside critical section to avoid race condition */ 
        pthread_mutex_lock(&(pCover->EventMutex));
        err = pthread_cond_broadcast(&(pCover->CondVar));
        pthread_mutex_unlock(&(pCover->EventMutex));
      }
    }
    if (err != E_OK)
    {
      err = E_FAIL;
    }
    return err;
}

/**
* Clear a flag (bitpattern).
*  
* /param pCover        pointer to flag structure.
* /param setptn        flag pattern
*
* /return E_OK
*         E_FAIL       Any error.
*/
EXPORT ER TRACE_clear_flag(TRACE_flg* pCover, U32 ClearPattern)
{
  ER rc = E_FAIL;
  if(NULL != pCover)
  {
    /*clear event*/
    rc = TRACE_set_FlagEvent(pCover, ClearPattern, FALSE);
  }
  if(E_OK != rc)
  {
    rc = E_FAIL;
  }
  return rc;
}

/*Time related API's required for Linux*/
LOCAL ER TRACE_get_time( struct timespec *p_timeout,  S32 tmout);
LOCAL ER TRACE_get_time( struct timespec *p_timeout,  S32 tmout)
{
    S32 sec = 0;
    clock_gettime( CLOCK_REALTIME, p_timeout);
    sec = (S32)(p_timeout->tv_nsec/(1000*1000) + tmout) / 1000;
    p_timeout->tv_sec+= sec;
    p_timeout->tv_nsec+=((1000*1000)*tmout)-(sec*1000*1000*1000);
    return E_OK;
}

/**
* Wait for a flg
*  
* /param pCover        pointer to flag structure.
* /param pAttr         pointer to flag attribute.
*
* /return E_OK
*         E_FAIL       Invalid parameter or mutex error.
*/
EXPORT ER TRACE_wait_flag(TRACE_flg *pCover, TRACE_flagAttr *pAttr )
{
    ER   err        = E_OK;
    ER   ret        = E_OK;
    U32  CurPattern = 0;
    BOOL exitWait = FALSE;

    if(!pCover )
    {
        return E_FAIL;
    }

    /* wait for 'event' */
    pthread_mutex_lock(&(pCover->EventMutex));

    /* check our pattern */
    while (1)
    {
        CurPattern = pCover->iflgptn;
        if((CurPattern & pAttr->waiptn) == 0)
        {
            if (pAttr->tmout== TMO_FEVR)
            {
                err = pthread_cond_wait( &(pCover->CondVar) , &(pCover->EventMutex) );
            }
            else
            {
                struct timespec timeout_chk;
                struct timespec timeout;
                TRACE_get_time( &timeout, pAttr->tmout);

                do
                {
                    err = pthread_cond_timedwait( &(pCover->CondVar) , &(pCover->EventMutex) ,&timeout);

                    if (err ==  ETIMEDOUT)
                    {

                        clock_gettime( CLOCK_REALTIME, &timeout_chk);


                        if( ( timeout_chk.tv_nsec >= timeout.tv_nsec ) && ( timeout_chk.tv_sec>=timeout.tv_sec ) )
                        {
                            exitWait = TRUE;
                            break;
                        }						
                    }
                    else
                    {
                        break;
                    }
                }while( 1 );
                if(FALSE != exitWait)
                {
                  break;
                }
            }
            if ( err != 0)
            {
                (void)pthread_mutex_unlock(&(pCover->EventMutex));
                return E_FAIL;
            }
        } /*  if((CurPattern & pAttr->waiptn) == 0) */
        CurPattern = pCover->iflgptn;
        /* compare for releasing */
        if( pAttr->wfmode & TWF_ORW )
        {
            if( CurPattern & pAttr->waiptn )
            {
                /* a bit was set -> break */
                pAttr->flg_ptn = CurPattern;
                break;
            }
        }
        else
        {
            if( (CurPattern & pAttr->waiptn ) == pAttr->waiptn )
            {
                /* a bit was set -> break */
                pAttr->flg_ptn = CurPattern;
                break;
            }
        }
    } /*End of while (1)*/
    
    if(TRUE != exitWait)
    {
      /* clear the bits  */ 
      if ( ( pAttr->wfmode & (TWF_CLR | TWF_BITCLR ) ) == TWF_CLR )
      {
          /* clear all bits */
          TRACE_set_FlagEvent_nolock( pCover,0xffffffff,FALSE);
      }

      if ( ( pAttr->wfmode & (TWF_CLR | TWF_BITCLR ) ) == TWF_BITCLR )
      {
          /* clear specified bit */
          TRACE_set_FlagEvent_nolock(pCover, pAttr->waiptn, FALSE);
      }
    }


    /* set timeout */
    if (err == ETIMEDOUT)
    {
        ret = ETIMEDOUT;
    }

    err = pthread_mutex_unlock(&(pCover->EventMutex));
    if(err != 0)
    {
      ret = E_FAIL;
    }
    return ret;
}



EXPORT ER TRACE_mutex_init(pthread_mutex_t* mutexLock)
{
	ER	err		=	E_OK;

	pthread_mutexattr_t mutexAttribute;

	memset(&mutexAttribute, 0, sizeof(mutexAttribute) );

	if(0 == (err = pthread_mutexattr_init(&mutexAttribute)))
	{
		if(0 == (err = pthread_mutexattr_setpshared(&mutexAttribute,
											PTHREAD_PROCESS_SHARED)) )
		{
			if(0 == (err = pthread_mutexattr_setrobust(&mutexAttribute, PTHREAD_MUTEX_ROBUST)))
			{
				if(0 == (err = pthread_mutex_init(mutexLock, &mutexAttribute)))
				{
					pthread_mutexattr_destroy(&mutexAttribute);
				}
			}
		}
	}
	return err;
}
/* PRQA: Lint Message 455: Mutex which is locked is only unlocked */
/*lint -save -e455 */
/* PRQA: Lint Message 454: Mutex which is locked is only unlocked */
/*lint -save -e454 */

EXPORT ER TRACE_mutex_lock(pthread_mutex_t* mutexLock)
{
	ER rc = pthread_mutex_lock(mutexLock);
	if(rc == EOWNERDEAD)
	{
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "%s failed with error= %d line=%d \n", __func__, rc,__LINE__);
		rc=pthread_mutex_consistent(mutexLock);
		if(rc)
			TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "%s failed with error= %d line=%d \n", __func__, rc,__LINE__);
	}
	else if(rc)
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "%s failed with error= %d line=%d \n", __func__, rc,__LINE__);
	return rc;
}

EXPORT ER TRACE_obtain_q_lock(pthread_mutex_t* q_lock)
{
	return TRACE_mutex_lock(q_lock);
}


EXPORT ER TRACE_mutex_unlock(pthread_mutex_t* mutexLock)
{
	ER rc = pthread_mutex_unlock(mutexLock);
	if(rc)
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "%s failed with error= %d\n", __func__, rc);
	return rc;
}

EXPORT ER TRACE_release_q_lock(pthread_mutex_t* q_lock)
{
	return TRACE_mutex_unlock(q_lock);
}

EXPORT ER TRACE_q_lock_init(pthread_mutex_t* q_lock)
{
	return TRACE_mutex_init(q_lock);
}
